import { MediatorProfileService } from '@/modules/mediator/profile/service';
import { getCurrentUser } from '@/modules/auth/actions';
import { notFound } from 'next/navigation';
import Link from 'next/link';
import { InquiryForm } from '@/components/case/InquiryForm';

const profileService = new MediatorProfileService();

export default async function PracticePage({
  params,
}: {
  params: { slug: string };
}) {
  let mediator;
  try {
    mediator = await profileService.getProfileBySlug(params.slug);
  } catch {
    notFound();
  }

  // Only show if published
  if (mediator.status !== 'PUBLISHED') {
    notFound();
  }

  // Check if user is logged in as client
  const user = await getCurrentUser();
  const isClient = user?.role === 'CLIENT';

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4 flex items-center justify-between">
          <Link href="/" className="flex items-center space-x-2">
            <div className="w-10 h-10 bg-primary-600 rounded-lg flex items-center justify-center">
              <span className="text-white font-bold text-xl">M8</span>
            </div>
            <span className="text-2xl font-bold text-gray-900">medi8</span>
          </Link>
          <div className="flex items-center space-x-4">
            <Link href="/practices" className="text-gray-600 hover:text-gray-900">
              ← Browse Mediators
            </Link>
            <Link href="/register/client" className="btn-primary">
              Book Consultation
            </Link>
          </div>
        </div>
      </header>

      {/* Content */}
      <main className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div className="bg-white rounded-lg shadow-lg p-8">
          {/* Header */}
          <div className="flex items-start gap-6 mb-8">
            {mediator.photoUrl && (
              <img
                src={mediator.photoUrl}
                alt={mediator.displayName || `${mediator.firstName} ${mediator.lastName}`}
                className="w-32 h-32 rounded-full object-cover"
              />
            )}
            
            <div className="flex-1">
              <h1 className="text-3xl font-bold mb-2">
                {mediator.displayName || `${mediator.firstName} ${mediator.lastName}`}
              </h1>
              
              {mediator.specializations.length > 0 && (
                <div className="flex flex-wrap gap-2 mb-4">
                  {mediator.specializations.map((spec) => (
                    <span
                      key={spec}
                      className="px-3 py-1 bg-primary-100 text-primary-800 rounded-full text-sm"
                    >
                      {spec}
                    </span>
                  ))}
                </div>
              )}
              
              <div className="flex items-center gap-4 text-sm text-gray-600">
                {mediator.yearsExperience && (
                  <span>{mediator.yearsExperience} years experience</span>
                )}
                {mediator.languages.length > 0 && (
                  <span>• {mediator.languages.join(', ')}</span>
                )}
              </div>
            </div>
          </div>

          {/* Bio */}
          {mediator.bio && (
            <div className="mb-8">
              <h2 className="text-xl font-semibold mb-3">About</h2>
              <p className="text-gray-700 whitespace-pre-line">{mediator.bio}</p>
            </div>
          )}

          {/* Certifications */}
          {mediator.certifications.length > 0 && (
            <div className="mb-8">
              <h2 className="text-xl font-semibold mb-3">Certifications & Qualifications</h2>
              <ul className="list-disc list-inside text-gray-700 space-y-1">
                {mediator.certifications.map((cert, i) => (
                  <li key={i}>{cert}</li>
                ))}
              </ul>
            </div>
          )}

          {/* Pricing */}
          {mediator.pricing && (
            <div className="mb-8">
              <h2 className="text-xl font-semibold mb-3">Pricing</h2>
              <div className="bg-gray-50 p-4 rounded-lg">
                {mediator.pricing.preMediationFee && (
                  <p className="mb-2">
                    <span className="font-medium">Pre-mediation consultation:</span>{' '}
                    ${mediator.pricing.preMediationFee.toString()}
                    {mediator.isGstRegistered && ' (inc GST)'}
                  </p>
                )}
                
                <p className="mb-2">
                  <span className="font-medium">Hourly rate:</span>{' '}
                  ${mediator.pricing.hourlyRate.toString()}/hour
                  {mediator.pricing.hourlyRateBasis === 'PER_PARTY' && ' per party'}
                  {mediator.isGstRegistered && ' (inc GST)'}
                </p>

                {mediator.pricing.minimumHours && (
                  <p className="text-sm text-gray-600 mb-2">
                    Minimum {mediator.pricing.minimumHours} hours
                  </p>
                )}

                {mediator.pricing.travelIncludedKm && (
                  <p className="text-sm text-gray-600 mb-2">
                    Travel: {mediator.pricing.travelIncludedKm}km included
                    {mediator.pricing.travelRatePerKm && 
                      `, $${mediator.pricing.travelRatePerKm.toString()}/km thereafter`
                    }
                  </p>
                )}

                {mediator.pricing.accommodationPolicy && (
                  <p className="text-sm text-gray-600 mb-2">
                    <span className="font-medium">Accommodation:</span> {mediator.pricing.accommodationPolicy}
                  </p>
                )}

                {mediator.pricing.extraCostsNotes && (
                  <p className="text-sm text-gray-600 mt-3 pt-3 border-t">
                    {mediator.pricing.extraCostsNotes}
                  </p>
                )}
              </div>
            </div>
          )}

          {/* Contact CTA / Inquiry Form */}
          {isClient ? (
            <div className="pt-6 border-t">
              <h2 className="text-xl font-semibold mb-4">Submit an Inquiry</h2>
              <InquiryForm mediatorId={mediator.id} />
            </div>
          ) : (
            <div className="text-center pt-6 border-t">
              <Link
                href="/register/client"
                className="inline-block bg-primary-600 text-white px-8 py-3 rounded-lg font-semibold hover:bg-primary-700 transition-colors"
              >
                Request Consultation
              </Link>
              <p className="text-sm text-gray-600 mt-3">
                Create a free account to contact {mediator.firstName}
              </p>
            </div>
          )}
        </div>
      </main>
    </div>
  );
}
